jQuery(document).ready(function ($) {
  // Auto-resize textareas with class .auto-resize (for settings prompts)
  function autoResizeTextarea(el) {
    el.style.height = "auto";
    el.style.height = el.scrollHeight + "px";
  }

  function initAutoResize(scope) {
    const $textareas = (scope ? $(scope) : $(document)).find(
      "textarea.auto-resize",
    );
    $textareas.each(function () {
      autoResizeTextarea(this);
    });
    $textareas.off("input.autoResize").on("input.autoResize", function () {
      autoResizeTextarea(this);
    });
  }

  // Initialize on load
  initAutoResize();

  // Also observe additions to prompts list and initialize on the new nodes
  const promptsList = document.getElementById("prompts-list");
  if (promptsList) {
    const observer = new MutationObserver(function (mutations) {
      for (const mutation of mutations) {
        if (mutation.type === "childList" && mutation.addedNodes.length) {
          mutation.addedNodes.forEach(function (node) {
            if (node.nodeType === 1) {
              initAutoResize(node);
            }
          });
        }
      }
    });
    observer.observe(promptsList, { childList: true, subtree: true });
  }

  // Make selectedImages and updateSelectedImagesDisplay available globally
  if (!window.danavarSelectedImages) {
    window.danavarSelectedImages = [];
    console.log("Created global danavarSelectedImages array");
  } else {
    console.log("Global danavarSelectedImages array already exists");
  }
  let selectedImages = window.danavarSelectedImages;
  let requestId = null;
  let statusCheckInterval = null;

  // WordPress Media Gallery
  $("#select-gallery-images").on("click", function () {
    openMediaGallery();
  });

  let frame;
  function openMediaGallery() {
    // Create a new frame if it doesn't exist
    if (!frame) {
      frame = wp.media({
        title: danava_i18n.select_images,
        button: {
          text: danava_i18n.select_images,
        },
        multiple: true,
        library: {
          type: "image",
        },
      });

      // When a file is selected, run a callback.
      frame.on("select", function () {
        const attachments = frame.state().get("selection").models;
        const remainingSlots = 3 - selectedImages.length;

        if (selectedImages.length >= 3) {
          Swal.fire({
            icon: "warning",
            title: danava_i18n.image_limit_title,
            text: danava_i18n.max_3_images,
            confirmButtonText: danava_i18n.understood,
          });
          return;
        }

        // Add newly selected images
        attachments.forEach(function (attachment) {
          if (selectedImages.length >= 3) {
            return;
          }
          const imageUrl =
            attachment.get("sizes") && attachment.get("sizes").full
              ? attachment.get("sizes").full.url
              : attachment.get("url");
          window.danavarSelectedImages.push({
            type: "gallery",
            id: attachment.get("id"),
            url: imageUrl,
            title: attachment.get("title"),
          });
        });

        if (attachments.length > remainingSlots) {
          Swal.fire({
            icon: "warning",
            title: danava_i18n.too_many_title,
            text: danava_i18n.max_3_images_detail,
            confirmButtonText: danava_i18n.understood,
          });
        }

        updateSelectedImagesDisplay();
        $("#selected-images-section")
          .get(0)
          .scrollIntoView({ behavior: "smooth" });
      });
    }

    frame.open();
  }

  function updateSelectedImagesDisplay() {
    console.log(
      "updateSelectedImagesDisplay called, images:",
      window.danavarSelectedImages,
    );
    const $grid = $("#selected-images-grid");
    const $noImages = $("#no-selected-images");

    $grid.empty();

    if (window.danavarSelectedImages.length === 0) {
      $noImages.removeClass("d-none");
      return;
    }

    $noImages.addClass("d-none");

    window.danavarSelectedImages.forEach((imageData, index) => {
      const $imageItem = $(`
                <div class="selected-image-item">
                    <img src="${imageData.url}" alt="Selected image ${index + 1}">
                    <button type="button" class="selected-image-remove" data-index="${index}">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `);

      $imageItem.find(".selected-image-remove").on("click", function () {
        removeSelectedImage($(this).data("index"));
      });

      $grid.append($imageItem);
    });
  }

  // Make updateSelectedImagesDisplay available globally
  window.danavaUpdateSelectedImagesDisplay = updateSelectedImagesDisplay;
  console.log("Global danavaUpdateSelectedImagesDisplay function set");

  function removeSelectedImage(index) {
    window.danavarSelectedImages.splice(index, 1);
    updateSelectedImagesDisplay();
  }

  // Prompt options
  $(".option-btn").on("click", function () {
    const value = $(this).data("value");
    $("#img-gen-prompt").val(value);
    $(".option-btn").removeClass("selected");
    $(this).addClass("selected");
  });

  // Form submission
  $("#main-submit-btn").on("click", function () {
    const $btn = $(this);
    const prompt = $("#img-gen-prompt").val().trim();

    if (!prompt) {
      Swal.fire({
        icon: "warning",
        title: danava_i18n.empty_description,
        text: danava_i18n.enter_description,
        confirmButtonText: danava_i18n.understood,
      });
      return;
    }
    if (window.danavarSelectedImages.length === 0) {
      Swal.fire({
        icon: "warning",
        title: danava_i18n.no_image_selected,
        text: danava_i18n.select_at_least_one,
        confirmButtonText: danava_i18n.understood,
      });
      return;
    }

    $btn
      .prop("disabled", true)
      .html('<i class="fas fa-spinner fa-spin"></i> ' + danava_i18n.sending);

    // Prepare form data
    const formData = new FormData();
    formData.append("action", "danava_submit_image_request");
    formData.append("nonce", danava_ajax.nonce);
    formData.append("prompt", prompt);

    // Add gallery images
    window.danavarSelectedImages.forEach((image) => {
      formData.append("images[]", image.url);
    });

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: formData,
      processData: false,
      contentType: false,
      success: function (response) {
        if (response.success) {
          requestId = response.data.request_id;
          showProcessingStatus();
          startStatusChecking();
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: response.data.message || danava_i18n.error_sending_request,
            confirmButtonText: danava_i18n.understood,
          });
          $btn.prop("disabled", false).html(danava_i18n.start_image_generation);
        }
      },
      error: function () {
        Swal.fire({
          icon: "error",
          title: danava_i18n.connection_error,
          text: danava_i18n.error_sending_request,
          confirmButtonText: danava_i18n.understood,
        });
        $btn.prop("disabled", false).html(danava_i18n.start_image_generation);
      },
    });
  });

  function showProcessingStatus() {
    $(".danava-image-editor").html(`
            <div class="processing-status text-center">
                <div class="spinner-border text-primary" style="width: 3rem; height: 3rem;" role="status">
                    <span class="sr-only">${danava_i18n.processing}</span>
                </div>
                <h4 class="mt-3">${danava_i18n.generating_image}</h4>
                <p>${danava_i18n.please_wait}</p>
                <div class="mt-4">
                    <button class="btn btn-secondary" onclick="location.reload()">${danava_i18n.back}</button>
                </div>
            </div>
        `);
  }

  function startStatusChecking() {
    statusCheckInterval = setInterval(checkStatus, 5000);
  }

  function checkStatus() {
    if (!requestId) return;

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_check_status",
        request_id: requestId,
        nonce: danava_ajax.nonce,
      },
      success: function (response) {
        if (response.success) {
          const status = response.data.status;
          if (status === "completed") {
            clearInterval(statusCheckInterval);
            showCompletedImage(response.data);
          } else if (status === "failed") {
            clearInterval(statusCheckInterval);
            showFailedStatus(response.data);
          }
          // Continue checking for pending/processing
        }
      },
      error: function () {
        // Continue checking on error
      },
    });
  }

  function saveToGallery($btn, imageUrl) {
    $btn.prop("disabled", true).text(danava_i18n.saving);

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_save_to_gallery",
        image_url: imageUrl,
        nonce: danava_ajax.nonce,
      },
      success: function (response) {
        if (response.success) {
          Swal.fire({
            icon: "success",
            title: danava_i18n.success,
            text: danava_i18n.image_saved_to_gallery,
            confirmButtonText: danava_i18n.great,
            timer: 2000,
          });
          $btn.text(danava_i18n.saved).addClass("btn-success");
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: response.data || danava_i18n.error_saving_image,
            confirmButtonText: danava_i18n.understood,
          });
          $btn.prop("disabled", false).text(danava_i18n.save_to_gallery);
        }
      },
      error: function (xhr, status, error) {
        Swal.fire({
          icon: "error",
          title: danava_i18n.error,
          text: danava_i18n.error_saving_image,
          confirmButtonText: danava_i18n.understood,
        });
        $btn.prop("disabled", false).text(danava_i18n.save_to_gallery);
      },
    });
  }

  function showCompletedImage(data) {
    const imageUrl =
      data.local_image_url || data.output_image_url || data.output_image;

    $(".danava-image-editor").html(`
            <div class="completed-status text-center">
                <div class="alert alert-success">
                    <h4><i class="fas fa-check-circle"></i> ${danava_i18n.image_generated_success}</h4>
                </div>
                <div class="image-result mb-4">
                    <img src="${imageUrl}" alt="Generated image" class="img-fluid rounded shadow" style="max-width: 500px;">
                </div>
                <div class="danava-action-buttons">
                    <button id="save-to-gallery-btn" class="danava-btn danava-btn-save" data-image-url="${imageUrl}">
                        <i class="fas fa-save"></i> ${danava_i18n.save_to_gallery}
                    </button>
                    <button id="use-in-product-btn" class="danava-btn danava-btn-use" data-image-url="${imageUrl}">
                        <i class="fas fa-shopping-cart"></i> ${danava_i18n.use_in_product}
                    </button>
                    <button class="danava-btn danava-btn-new" onclick="location.reload()">
                        <i class="fas fa-plus"></i> ${danava_i18n.new_request}
                    </button>
                </div>
            </div>
        `);

    // Attach save to gallery handler
    $("#save-to-gallery-btn").on("click", function () {
      saveToGallery($(this), $(this).data("imageUrl"));
    });

    // Attach use in product handler
    $("#use-in-product-btn").on("click", function () {
      openProductSelectionModal($(this).data("imageUrl"));
    });
  }

  function showFailedStatus(data) {
    $(".danava-image-editor").html(`
            <div class="failed-status text-center">
                <div class="alert alert-danger">
                    <h4><i class="fas fa-exclamation-triangle"></i> ${danava_i18n.error_generating_image}</h4>
                    <p>${data.error_message || danava_i18n.unknown_error}</p>
                </div>
                <div>
                    <button class="btn btn-primary" onclick="location.reload()">${danava_i18n.try_again}</button>
                </div>
            </div>
        `);
  }

  // WooCommerce Product Integration
  let selectedProductId = null;
  let selectedProductGalleryImages = [];
  let galleryOrderModified = false;

  function openProductSelectionModal(imageUrl) {
    // Create modal HTML
    const modalHtml = `
      <div id="product-selection-modal" class="danava-modal" style="display: none;">
        <div class="danava-modal-content">
          <div class="danava-modal-header">
            <h3>${danava_i18n.select_product}</h3>
            <button class="danava-modal-close" onclick="jQuery('#product-selection-modal').fadeOut()">
              <i class="fas fa-times"></i>
            </button>
          </div>
          <div class="danava-modal-body">
            <div class="product-search-box mb-3">
              <input type="text" id="product-search-input" class="form-control" placeholder="${danava_i18n.search_product}">
            </div>
            <div id="products-list" class="products-list">
              <div class="text-center py-4">
                <i class="fas fa-spinner fa-spin fa-2x"></i>
                <p>${danava_i18n.loading_products}</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    `;

    // Remove existing modal if any
    $("#product-selection-modal").remove();

    // Append modal to body
    $("body").append(modalHtml);

    // Show modal
    $("#product-selection-modal").fadeIn();

    // Load products
    loadProducts("", imageUrl);

    // Search handler with debounce
    let searchTimeout;
    $("#product-search-input").on("input", function () {
      clearTimeout(searchTimeout);
      const searchQuery = $(this).val();
      searchTimeout = setTimeout(function () {
        loadProducts(searchQuery, imageUrl);
      }, 500);
    });
  }
  // Make function globally available for gallery.js
  window.openProductSelectionModal = openProductSelectionModal;

  function loadProducts(searchQuery, imageUrl) {
    const $productsList = $("#products-list");

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_search_products",
        nonce: danava_ajax.nonce,
        search: searchQuery,
        page: 1,
      },
      success: function (response) {
        if (response.success && response.data.products) {
          displayProducts(response.data.products, imageUrl);
        } else {
          $productsList.html(
            `<div class="text-center py-4"><p>${danava_i18n.no_product_found}</p></div>`,
          );
        }
      },
      error: function () {
        $productsList.html(
          `<div class="text-center py-4 text-danger"><p>${danava_i18n.error_loading_products}</p></div>`,
        );
      },
    });
  }

  function displayProducts(products, imageUrl) {
    const $productsList = $("#products-list");
    $productsList.empty();

    if (products.length === 0) {
      $productsList.html(
        `<div class="text-center py-4"><p>${danava_i18n.no_product_found}</p></div>`,
      );
      return;
    }

    products.forEach(function (product) {
      const thumbnailImg = product.thumbnail
        ? `<img src="${product.thumbnail}" alt="${product.title}">`
        : '<i class="fas fa-image fa-2x"></i>';

      const $productItem = $(`
        <div class="product-item" data-product-id="${product.id}">
          <div class="product-thumbnail">
            ${thumbnailImg}
          </div>
          <div class="product-info">
            <h4>${product.title}</h4>
            ${product.sku ? `<p class="product-sku">SKU: ${product.sku}</p>` : ""}
            <p class="product-images-count">
              <i class="fas fa-images"></i> ${product.total_images} ${danava_i18n.images_text}
            </p>
          </div>
          <div class="product-actions">
            <button class="btn btn-sm btn-primary select-product-btn">
              ${danava_i18n.select}
            </button>
          </div>
        </div>
      `);

      $productItem.find(".select-product-btn").on("click", function () {
        selectedProductId = product.id;
        loadProductImages(product.id, imageUrl);
      });

      $productsList.append($productItem);
    });
  }

  function loadProductImages(productId, imageUrl) {
    const $modalBody = $("#product-selection-modal .danava-modal-body");

    $modalBody.html(`
      <div class="text-center py-4">
        <i class="fas fa-spinner fa-spin fa-2x"></i>
        <p>${danava_i18n.loading_product_images}</p>
      </div>
    `);

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_get_product_images",
        nonce: danava_ajax.nonce,
        product_id: productId,
      },
      success: function (response) {
        if (response.success) {
          selectedProductGalleryImages = response.data.images || [];
          displayProductImagesOptions(
            response.data.product_title,
            response.data.images,
            imageUrl,
          );
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: danava_i18n.error_loading_product_images,
            confirmButtonText: danava_i18n.understood,
          });
        }
      },
      error: function () {
        Swal.fire({
          icon: "error",
          title: danava_i18n.error,
          text: danava_i18n.error_loading_product_images,
          confirmButtonText: danava_i18n.understood,
        });
      },
    });
  }

  function displayProductImagesOptions(
    productTitle,
    existingImages,
    newImageUrl,
  ) {
    const $modalBody = $("#product-selection-modal .danava-modal-body");

    const galleryImagesHtml = existingImages
      .map(
        (img, index) => `
      <div class="gallery-image-item ${img.is_featured ? "featured-image" : ""}"
           data-image-id="${img.id}"
           data-order="${index}">
        <img src="${img.thumbnail}" alt="${img.title}">
        ${img.is_featured ? `<span class="badge badge-primary">${danava_i18n.main_image_badge}</span>` : ""}
        <div class="image-order-controls">
          <button class="btn btn-sm move-up" ${index === 0 ? "disabled" : ""}>
            <i class="fas fa-chevron-up"></i>
          </button>
          <button class="btn btn-sm move-down" ${index === existingImages.length - 1 ? "disabled" : ""}>
            <i class="fas fa-chevron-down"></i>
          </button>
        </div>
      </div>
    `,
      )
      .join("");

    $modalBody.html(`
      <div class="product-image-options">
        <div class="mb-3">
          <button class="btn btn-sm btn-secondary" id="back-to-products">
            <i class="fas fa-arrow-right"></i> ${danava_i18n.back_to_products}
          </button>
        </div>

        <h4 class="mb-3">${productTitle}</h4>

        <div class="new-image-preview mb-4">
          <h5>${danava_i18n.new_image}</h5>
          <img src="${newImageUrl}" alt="New image" style="max-width: 200px;" class="rounded">
        </div>

        <div class="image-action-options mb-4">
          <h5>${danava_i18n.how_to_use_image}</h5>
          <div class="form-check">
            <input class="form-check-input" type="radio" name="imageAction" id="action-main" value="main" checked>
            <label class="form-check-label" for="action-main">
              ${danava_i18n.set_as_main}
            </label>
          </div>
          <div class="form-check">
            <input class="form-check-input" type="radio" name="imageAction" id="action-gallery" value="gallery">
            <label class="form-check-label" for="action-gallery">
              ${danava_i18n.add_to_gallery}
            </label>
          </div>
          <div class="form-check">
            <input class="form-check-input" type="radio" name="imageAction" id="action-both" value="both">
            <label class="form-check-label" for="action-both">
              ${danava_i18n.set_as_main_and_gallery}
            </label>
          </div>
        </div>

        ${
          existingImages.length > 0
            ? `
          <div class="existing-gallery mb-4">
            <h5>${danava_i18n.existing_product_images}</h5>
            <div id="gallery-images-sortable" class="gallery-images-grid">
              ${galleryImagesHtml}
            </div>
            <p class="text-muted small">
              <i class="fas fa-info-circle"></i>
              ${danava_i18n.reorder_info}
            </p>
          </div>
        `
            : ""
        }

        <div class="text-center">
          <button id="confirm-product-image-btn" class="btn btn-success btn-lg">
            <i class="fas fa-check"></i> ${danava_i18n.confirm_and_apply}
          </button>
        </div>
      </div>
    `);

    // Back button handler
    $("#back-to-products").on("click", function () {
      openProductSelectionModal(newImageUrl);
    });

    // Reset gallery order flag when loading new product
    galleryOrderModified = false;

    // Gallery image reordering
    $(".move-up").on("click", function () {
      const $item = $(this).closest(".gallery-image-item");
      const $prev = $item.prev();
      if ($prev.length) {
        $item.insertBefore($prev);
        updateImageOrder();
      }
    });

    $(".move-down").on("click", function () {
      const $item = $(this).closest(".gallery-image-item");
      const $next = $item.next();
      if ($next.length) {
        $item.insertAfter($next);
        updateImageOrder();
      }
    });

    // Confirm button handler
    $("#confirm-product-image-btn").on("click", function () {
      applyImageToProduct(newImageUrl);
    });
  }

  function updateImageOrder() {
    const $items = $("#gallery-images-sortable .gallery-image-item");
    $items.each(function (index) {
      $(this).attr("data-order", index);

      // Update button states
      $(this)
        .find(".move-up")
        .prop("disabled", index === 0);
      $(this)
        .find(".move-down")
        .prop("disabled", index === $items.length - 1);
    });

    // Mark that gallery order was modified
    galleryOrderModified = true;
  }

  function applyImageToProduct(imageUrl) {
    const action = $('input[name="imageAction"]:checked').val();
    const $btn = $("#confirm-product-image-btn");

    // Get reordered gallery images only if modified
    const galleryOrder = [];
    if (galleryOrderModified) {
      $("#gallery-images-sortable .gallery-image-item").each(function () {
        const imageId = parseInt($(this).attr("data-image-id"));
        if (!isNaN(imageId)) {
          galleryOrder.push(imageId);
        }
      });
    }

    $btn
      .prop("disabled", true)
      .html('<i class="fas fa-spinner fa-spin"></i> ' + danava_i18n.applying);

    const ajaxData = {
      action: "danava_apply_image_to_product",
      nonce: danava_ajax.nonce,
      product_id: selectedProductId,
      image_url: imageUrl,
      image_action: action,
    };

    // Only send gallery_order if it was modified and has items
    if (galleryOrderModified && galleryOrder.length > 0) {
      ajaxData.gallery_order = JSON.stringify(galleryOrder);
    }

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: ajaxData,
      success: function (response) {
        if (response.success) {
          Swal.fire({
            icon: "success",
            title: danava_i18n.success,
            text: danava_i18n.image_added_to_product,
            confirmButtonText: danava_i18n.great,
          }).then(() => {
            $("#product-selection-modal").fadeOut();

            // Reset gallery order flag
            galleryOrderModified = false;

            // Optional: Show product link
            if (response.data && response.data.product_url) {
              Swal.fire({
                icon: "question",
                title: danava_i18n.view_product,
                text: danava_i18n.view_product_question,
                showCancelButton: true,
                confirmButtonText: danava_i18n.yes_view,
                cancelButtonText: danava_i18n.no,
              }).then((result) => {
                if (result.isConfirmed) {
                  window.open(response.data.product_url, "_blank");
                }
              });
            }
          });
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: response.data || danava_i18n.error_applying_image,
            confirmButtonText: danava_i18n.understood,
          });
          $btn
            .prop("disabled", false)
            .html('<i class="fas fa-check"></i> ' + danava_i18n.confirm_and_apply);
        }
      },
      error: function () {
        Swal.fire({
          icon: "error",
          title: danava_i18n.connection_error,
          text: danava_i18n.error_applying_image,
          confirmButtonText: danava_i18n.understood,
        });
        $btn
          .prop("disabled", false)
          .html('<i class="fas fa-check"></i> ' + danava_i18n.confirm_and_apply);
      },
    });
  }
});
