jQuery(document).ready(function ($) {
  let currentPage = 1;
  let loadingImages = false;
  const modal = $("#image-details-modal");
  const closeBtn = $(".danava-modal-close");
  let touchStartY = 0;
  let touchEndY = 0;

  // Initial load
  loadGalleryImages();

  // Load more button click
  $("#load-more-images").on("click", function () {
    if (!loadingImages) {
      currentPage++;
      loadGalleryImages();
    }
  });

  // Close modal function
  function closeModal() {
    modal.hide();
    $("body").css("overflow", "auto");
  }

  // Close modal
  closeBtn.on("click", function () {
    closeModal();
  });

  $(window).on("click", function (event) {
    if (event.target === modal[0]) {
      closeModal();
    }
  });

  // Escape key to close modal
  $(document).on("keydown", function (e) {
    if (e.key === "Escape" && modal.is(":visible")) {
      closeModal();
    }
  });

  // Touch events for mobile swipe to close
  modal.on("touchstart", function (e) {
    if (e.target === modal[0]) {
      touchStartY = e.touches[0].clientY;
    }
  });

  modal.on("touchend", function (e) {
    if (e.target === modal[0]) {
      touchEndY = e.changedTouches[0].clientY;
      handleSwipe();
    }
  });

  function handleSwipe() {
    const swipeDistance = touchStartY - touchEndY;
    // If swipe down more than 100px, close modal
    if (swipeDistance < -100) {
      closeModal();
    }
  }

  function loadGalleryImages() {
    loadingImages = true;
    const $loadMoreBtn = $("#load-more-images");
    $loadMoreBtn.text(danava_i18n.loading).prop("disabled", true);

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_load_gallery",
        nonce: danava_ajax.nonce,
        page: currentPage,
      },
      success: function (response) {
        if (response.success) {
          renderGalleryItems(response.data.items);
          $loadMoreBtn.toggle(response.data.has_more);
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text:
              danava_i18n.error_loading_images + " " + (response.data || "Unknown error"),
            confirmButtonText: danava_i18n.understood,
          });
        }
      },
      error: function (xhr, status, error) {
        Swal.fire({
          icon: "error",
          title: danava_i18n.connection_error,
          text: danava_i18n.error_connecting_to_server + " " + error,
          confirmButtonText: danava_i18n.understood,
        });
      },
      complete: function () {
        loadingImages = false;
        $loadMoreBtn.text(danava_i18n.load_more).prop("disabled", false);
      },
    });
  }

  function renderGalleryItems(items) {
    const $grid = $("#danava-gallery-grid");

    items.forEach((item) => {
      const $item = $(`
                <div class="gallery-item" data-id="${item.id}">
                    <img src="${item.output_image_thumbnail_url}" alt="Generated image">
                    <div class="gallery-item-info">
                        <div class="gallery-item-date">${formatDate(item.created_at)}</div>
                    </div>
                </div>
            `);

      $item.on("click", function () {
        loadImageDetails(item.id);
      });

      $grid.append($item);
    });
  }

  function loadImageDetails(id) {
    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_get_image_details",
        nonce: danava_ajax.nonce,
        id: id,
      },
      success: function (response) {
        if (response.success) {
          showImageDetails(response.data);
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: danava_i18n.error_loading_image_details,
            confirmButtonText: danava_i18n.understood,
          });
        }
      },
      error: function () {
        Swal.fire({
          icon: "error",
          title: danava_i18n.error,
          text: danava_i18n.error_connecting_to_server_short,
          confirmButtonText: danava_i18n.understood,
        });
      },
    });
  }

  function showImageDetails(data) {
    const $modal = $("#image-details-modal");
    const $inputContainer = $modal.find(".input-images-container");
    const $outputContainer = $modal.find(".output-image-container");
    const $modalContent = $modal.find(".danava-modal-content");

    // Clear containers
    $inputContainer.empty();
    $outputContainer.empty();

    // Prevent body scroll on mobile
    $("body").css("overflow", "hidden");

    // Reset scroll position
    $modalContent.scrollTop(0);

    // Add input images
    if (data.input_image) {
      $inputContainer.append(
        `<img src="${data.input_image}" alt="Input image 1">`,
      );
    }
    if (data.second_input_image) {
      $inputContainer.append(
        `<img src="${data.second_input_image}" alt="Input image 2">`,
      );
    }
    if (data.third_input_image) {
      $inputContainer.append(
        `<img src="${data.third_input_image}" alt="Input image 3">`,
      );
    }

    // Add output image
    $outputContainer.append(
      `<img src="${data.output_image}" alt="Output image">`,
    );

    // Update details
    $modal
      .find(".prompt-info")
      .html(`<strong>${danava_i18n.description}</strong> ${data.prompt}`);
    $modal
      .find(".date-info")
      .html(`<strong>${danava_i18n.created_date}</strong> ${formatDate(data.created_at)}`);
    $modal.find(".model-info").html(`<strong>${danava_i18n.model}</strong> ${data.model}`);

    // Get the output image URL (use the best quality available)
    const outputImageUrl =
      data.local_image_url || data.output_image_url || data.output_image;

    // Setup save to gallery button
    $modal
      .find(".save-to-gallery")
      .off("click")
      .on("click", function () {
        const $btn = $(this);
        $btn.prop("disabled", true).text(danava_i18n.saving);

        // Use existing save to gallery functionality
        $.ajax({
          url: danava_ajax.ajax_url,
          type: "POST",
          data: {
            action: "danava_save_to_gallery",
            nonce: danava_ajax.nonce,
            image_url: outputImageUrl,
          },
          success: function (response) {
            if (response.success) {
              Swal.fire({
                icon: "success",
                title: danava_i18n.success,
                text: danava_i18n.image_saved_to_gallery,
                confirmButtonText: danava_i18n.great,
                timer: 2000,
              });
            } else {
              Swal.fire({
                icon: "error",
                title: danava_i18n.error,
                text: danava_i18n.error_saving_image,
                confirmButtonText: danava_i18n.understood,
              });
            }
          },
          error: function () {
            Swal.fire({
              icon: "error",
              title: danava_i18n.error,
              text: danava_i18n.error_connecting_to_server_short,
              confirmButtonText: danava_i18n.understood,
            });
          },
          complete: function () {
            $btn.prop("disabled", false).text(danava_i18n.save_to_gallery);
          },
        });
      });

    // Setup use in product button
    $modal
      .find(".use-in-product")
      .attr("data-image-url", outputImageUrl)
      .off("click")
      .on("click", function () {
        const imageUrl = $(this).attr("data-image-url");

        // Close the gallery modal properly
        closeModal();

        // Check if the openProductSelectionModal function exists (from admin.js)
        if (typeof window.openProductSelectionModal === "function") {
          window.openProductSelectionModal(imageUrl);
        } else {
          // If admin.js is not loaded, we need to load it or show an error
          Swal.fire({
            icon: "info",
            title: danava_i18n.attention,
            text: danava_i18n.use_new_image_page,
            confirmButtonText: danava_i18n.understood,
          });
        }
      });

    // Show modal
    $modal.show();

    // Add image zoom functionality for mobile
    $modal.find("img").on("click", function (e) {
      e.stopPropagation();
      const $img = $(this);
      if ($img.hasClass("zoomed")) {
        $img.removeClass("zoomed");
      } else {
        $modal.find("img").removeClass("zoomed");
        $img.addClass("zoomed");
      }
    });
  }

  function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString("fa-IR", {
      year: "numeric",
      month: "long",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  }
});
