jQuery(document).ready(function ($) {
  let searchTimeout = null;
  let currentProductsPage = 1;
  let selectedProductImages = [];
  let currentProductId = null;

  // Wait for global variables to be ready
  function waitForGlobals(callback) {
    if (
      window.danavarSelectedImages &&
      window.danavaUpdateSelectedImagesDisplay
    ) {
      callback();
    } else {
      console.log("Waiting for global variables...");
      setTimeout(function () {
        waitForGlobals(callback);
      }, 100);
    }
  }

  // Initialize when globals are ready
  waitForGlobals(function () {
    console.log("Global variables are ready!");
    initWooCommerceSelector();
  });

  function initWooCommerceSelector() {
    // Open WooCommerce Product Selector Modal
    $("#select-woo-products").on("click", function () {
      $("#woo-product-selector-modal").show();
      loadProducts("", 1);
    });

    // Close Product Selector Modal
    $(".woo-modal-close, #close-product-modal").on("click", function () {
      $("#woo-product-selector-modal").hide();
    });

    // Close Images Modal
    $(".woo-images-modal-close, #cancel-image-selection").on(
      "click",
      function () {
        $("#woo-images-modal").hide();
      },
    );

    // Close modal when clicking outside
    $(window).on("click", function (event) {
      if (event.target.id === "woo-product-selector-modal") {
        $("#woo-product-selector-modal").hide();
      }
      if (event.target.id === "woo-images-modal") {
        $("#woo-images-modal").hide();
      }
    });

    // Real-time search
    $("#woo-product-search").on("input", function () {
      const searchTerm = $(this).val();

      // Clear previous timeout
      if (searchTimeout) {
        clearTimeout(searchTimeout);
      }

      // Show search spinner
      $(".woo-search-icon").hide();
      $(".woo-search-spinner").addClass("active");

      // Set new timeout for search
      searchTimeout = setTimeout(function () {
        currentProductsPage = 1;
        loadProducts(searchTerm, 1);
      }, 500); // 500ms delay for better UX
    });

    // Refresh available slots when modal opens
    $("#select-woo-products").on("click", function () {
      // Clear previous selections from product modal
      selectedProductImages = [];
    });

    // Clear product search
    $("#clear-product-search").on("click", function () {
      $("#woo-product-search").val("");
      currentProductsPage = 1;
      loadProducts("", 1);
    });

    // Confirm Image Selection
    $("#confirm-image-selection").on("click", function () {
      console.log(
        "Confirm clicked, selectedProductImages:",
        selectedProductImages,
      );

      if (selectedProductImages.length === 0) {
        Swal.fire({
          icon: "warning",
          title: danava_i18n.no_image_selected,
          text: danava_i18n.select_at_least_one,
          confirmButtonText: danava_i18n.understood,
        });
        return;
      }

      console.log("Global array before:", window.danavarSelectedImages);

      // Add selected images to main selected images array (global)
      if (window.danavarSelectedImages) {
        selectedProductImages.forEach((img) => {
          if (window.danavarSelectedImages.length < 3) {
            console.log("Adding image to global array:", img);
            window.danavarSelectedImages.push(img);
          }
        });

        console.log("Global array after:", window.danavarSelectedImages);

        // Update the display using global function
        if (window.danavaUpdateSelectedImagesDisplay) {
          console.log("Calling updateSelectedImagesDisplay");
          window.danavaUpdateSelectedImagesDisplay();
        } else {
          console.error(
            "danavaUpdateSelectedImagesDisplay function not found!",
          );
        }
      } else {
        console.error("danavarSelectedImages global array not found!");
      }

      // Reset and close modals
      selectedProductImages = [];
      $("#woo-images-modal").hide();
      $("#woo-product-selector-modal").hide();

      // Scroll to selected images section
      $("#selected-images-section")
        .get(0)
        .scrollIntoView({ behavior: "smooth" });

      // Show success message
      showSuccessMessage(danava_i18n.images_added_success);
    });
  } // End of initWooCommerceSelector

  // Load WooCommerce Products
  function loadProducts(searchTerm = "", page = 1) {
    const $grid = $("#woo-products-grid");
    const $loading = $("#woo-products-loading");
    const $noProducts = $("#woo-no-products");

    if (page === 1) {
      $grid.empty();
    }

    $loading.show();
    $noProducts.hide();

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_search_products",
        nonce: danava_ajax.nonce,
        search: searchTerm,
        page: page,
      },
      success: function (response) {
        // Hide search spinner
        $(".woo-search-spinner").removeClass("active");
        $(".woo-search-icon").show();
        $loading.hide();

        if (response.success) {
          const products = response.data.products;

          if (products.length === 0 && page === 1) {
            $noProducts.show();
            return;
          }

          renderProducts(products);
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text:
              danava_i18n.error_loading_products + ": " + (response.data || danava_i18n.unknown_error),
            confirmButtonText: danava_i18n.understood,
          });
        }
      },
      error: function (xhr, status, error) {
        $(".woo-search-spinner").removeClass("active");
        $(".woo-search-icon").show();
        $loading.hide();
        Swal.fire({
          icon: "error",
          title: danava_i18n.connection_error,
          text: danava_i18n.error_connecting_to_server + ": " + error,
          confirmButtonText: danava_i18n.understood,
        });
      },
    });
  }

  // Render Products Grid
  function renderProducts(products) {
    const $grid = $("#woo-products-grid");

    products.forEach((product) => {
      const imagesCountText = product.total_images + " " + danava_i18n.images_text;

      let imageHtml;
      if (product.thumbnail) {
        imageHtml = `<img src="${product.thumbnail}" alt="${product.title}" class="woo-product-image">`;
      } else {
        imageHtml = `<div class="woo-product-image" style="display: flex; align-items: center; justify-content: center; background: #f0f0f0; width: 100%; height: 100%;">
          <svg width="80" height="80" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M21 19V5C21 3.9 20.1 3 19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19ZM8.5 13.5L11 16.51L14.5 12L19 18H5L8.5 13.5Z" fill="#ccc"/>
          </svg>
        </div>`;
      }

      const $productCard = $(`
                <div class="woo-product-card" data-product-id="${product.id}">
                    <div class="woo-product-badge">${danava_i18n.selected_badge}</div>
                    <div class="woo-product-image-wrapper">
                        ${imageHtml}
                    </div>
                    <div class="woo-product-info">
                        <h4 class="woo-product-title">${product.title}</h4>
                        <div class="woo-product-meta">
                            <div class="woo-product-price">${product.price}</div>
                            <div class="woo-product-images-count">
                                <i class="fas fa-images"></i>
                                ${imagesCountText}
                            </div>
                        </div>
                    </div>
                </div>
            `);

      $productCard.on("click", function () {
        const productId = $(this).data("product-id");
        openProductImages(productId);
      });

      $grid.append($productCard);
    });
  }

  // Open Product Images Modal
  function openProductImages(productId) {
    currentProductId = productId;
    $("#woo-images-modal").show();
    $("#woo-images-grid").html(
      '<div class="woo-loading"><i class="fas fa-spinner fa-spin"></i><p>' + danava_i18n.loading_product_images + '</p></div>',
    );

    $.ajax({
      url: danava_ajax.ajax_url,
      type: "POST",
      data: {
        action: "danava_get_product_images",
        nonce: danava_ajax.nonce,
        product_id: productId,
      },
      success: function (response) {
        if (response.success) {
          const data = response.data;
          $("#woo-images-modal-title").text(data.product_title);
          renderProductImages(data.images);
          updateSelectedImagesCount();
        } else {
          Swal.fire({
            icon: "error",
            title: danava_i18n.error,
            text: danava_i18n.error_loading_product_images,
            confirmButtonText: danava_i18n.understood,
          });
          $("#woo-images-modal").hide();
        }
      },
      error: function () {
        Swal.fire({
          icon: "error",
          title: danava_i18n.error,
          text: danava_i18n.error_connecting_to_server_short,
          confirmButtonText: danava_i18n.understood,
        });
        $("#woo-images-modal").hide();
      },
    });
  }

  // Render Product Images
  function renderProductImages(images) {
    const $grid = $("#woo-images-grid");
    $grid.empty();

    if (images.length === 0) {
      $grid.html(
        '<div class="woo-no-products"><i class="fas fa-image"></i><h3>' + danava_i18n.no_images_found + '</h3><p>' + danava_i18n.product_no_images + '</p></div>',
      );
      return;
    }

    images.forEach((image) => {
      const isFeaturedBadge = image.is_featured
        ? '<span style="position: absolute; top: 8px; left: 8px; background: #28a745; color: white; padding: 2px 8px; border-radius: 4px; font-size: 10px; z-index: 2;">' + danava_i18n.main_badge + '</span>'
        : "";

      const $imageItem = $(`
                <div class="woo-image-item" data-image-id="${image.id}" data-image-url="${image.url}">
                    ${isFeaturedBadge}
                    <div class="woo-image-checkbox">
                        <i class="fas fa-check"></i>
                    </div>
                    <img src="${image.medium}" alt="${image.title || "Product image"}">
                </div>
            `);

      $imageItem.on("click", function () {
        toggleImageSelection($(this));
      });

      $grid.append($imageItem);
    });
  }

  // Toggle Image Selection
  function toggleImageSelection($imageItem) {
    const imageId = $imageItem.data("image-id");
    const imageUrl = $imageItem.data("image-url");

    console.log("Toggling image:", imageId, imageUrl);

    // Check if already selected
    const existingIndex = selectedProductImages.findIndex(
      (img) => img.id === imageId,
    );

    if (existingIndex !== -1) {
      // Deselect
      console.log("Deselecting image");
      selectedProductImages.splice(existingIndex, 1);
      $imageItem.removeClass("selected");
    } else {
      // Check if we can add more (max 3 total images)
      const currentSelectedCount = window.danavarSelectedImages
        ? window.danavarSelectedImages.length
        : 0;
      const totalSelected = currentSelectedCount + selectedProductImages.length;

      console.log("Current selected count:", currentSelectedCount);
      console.log("Total would be:", totalSelected);

      if (totalSelected >= 3) {
        Swal.fire({
          icon: "warning",
          title: danava_i18n.image_limit_title,
          text: danava_i18n.max_3_remove_first,
          confirmButtonText: danava_i18n.understood,
        });
        return;
      }

      // Select
      const imageData = {
        id: imageId,
        url: imageUrl,
        type: "woocommerce",
        product_id: currentProductId,
      };
      console.log("Selecting image:", imageData);
      selectedProductImages.push(imageData);
      $imageItem.addClass("selected");
    }

    console.log("selectedProductImages now:", selectedProductImages);
    updateSelectedImagesCount();
  }

  // Update Selected Images Count
  function updateSelectedImagesCount() {
    const count = selectedProductImages.length;
    const currentSelectedCount = window.danavarSelectedImages
      ? window.danavarSelectedImages.length
      : 0;
    const maxCount = 3 - currentSelectedCount;
    $("#woo-selected-images-count").html(
      `<strong>${count}</strong> ${danava_i18n.from} <strong>${maxCount}</strong> ${danava_i18n.image_selected}`,
    );

    // Enable/disable confirm button
    $("#confirm-image-selection").prop("disabled", count === 0);
  }

  // Show Success Message
  function showSuccessMessage(message) {
    Swal.fire({
      icon: "success",
      title: danava_i18n.success,
      text: message,
      confirmButtonText: danava_i18n.great,
      timer: 2000,
      toast: true,
      position: "top-end",
      showConfirmButton: false,
    });
  }
});
