<?php
/**
 * Danava API Handler Class
 *
 * Handles sending product images to Danava API.
 */

if (!defined("ABSPATH")) {
    exit();
}

class Danava_API_Handler
{
    private $api_key;
    private $api_url;
    private $model;

    public function __construct()
    {
        add_action("init", [$this, "init"]);
        add_action("wp_ajax_danava_submit_image_request", [
            $this,
            "ajax_submit_image_request",
        ]);
        add_action("wp_ajax_danava_check_status", [$this, "ajax_check_status"]);
        add_action("wp_ajax_danava_save_to_gallery", [
            $this,
            "ajax_save_to_gallery",
        ]);
    }

    public function init()
    {
        $this->api_key = get_option("danava_api_key");
        // Use saved API URL; fall back to current default endpoint used in settings
        $this->api_url = get_option(
            "danava_api_url",
            "https://danava-ai.ir/ai-support/api/v2/image-generator-requests/",
        );

        // Normalize/validate model against accepted values
        $saved_model = get_option("danava_model", "gemini");
        $allowed_models = ["gemini"];
        $this->model = in_array($saved_model, $allowed_models, true)
            ? $saved_model
            : "gemini";
    }

    /**
     * AJAX: Submit image request
     */
    public function ajax_submit_image_request()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_die("Unauthorized");
        }

        if (empty($this->api_key) || empty($this->api_url)) {
            wp_send_json_error(["message" => "API settings not configured"]);
        }

        if (!isset($_POST["prompt"])) {
            wp_send_json_error(["message" => "No prompt provided"]);
        }

        $prompt = sanitize_text_field(wp_unslash($_POST["prompt"]));

        // Prepare data
        $data = [
            "model" => $this->model,
            "prompt" => $prompt,
        ];

        // Add up to 3 input images as base64 using API-expected field names
        if (isset($_POST["images"]) && is_array($_POST["images"])) {
            $images = array_map("esc_url_raw", wp_unslash($_POST["images"]));

            foreach ($images as $index => $image_url) {
                if ($index > 2) {
                    // limit to 3 images
                    break;
                }

                // Convert URL to local file path
                $uploads_dir = wp_upload_dir();
                $base_url = $uploads_dir["baseurl"];
                $base_dir = $uploads_dir["basedir"];

                // Try to convert uploads URL to local path
                if (strpos($image_url, $base_url) !== false) {
                    $local_path = str_replace($base_url, $base_dir, $image_url);
                } else {
                    // Fallback: try replacing site URL with ABSPATH
                    $site_url = get_site_url();
                    $local_path = str_replace(
                        $site_url,
                        untrailingslashit(ABSPATH),
                        $image_url,
                    );
                }

                if (file_exists($local_path)) {
                    $image_data = file_get_contents($local_path);
                    if ($image_data !== false) {
                        $base64 = base64_encode($image_data);

                        // Map index to API field names
                        if ($index === 0) {
                            $data["image1"] = $base64;
                        } elseif ($index === 1) {
                            $data["image2"] = $base64;
                        } elseif ($index === 2) {
                            $data["image3"] = $base64;
                        }
                    }
                }
            }
        }

        // Send to API as JSON
        $response = wp_remote_post($this->api_url, [
            "body" => json_encode($data),
            "headers" => [
                "Content-Type" => "application/json",
                "X-API-Key" => $this->api_key,
            ],
            "timeout" => 30,
        ]);

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            wp_send_json_error([
                "message" => "API request failed: " . $error_message,
            ]);
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code !== 201) {
            wp_send_json_error(["message" => "API error: " . $response_body]);
        }

        $result = json_decode($response_body, true);
        if (is_numeric($result) && $result > 0) {
            // API returns just the ID as a number
            wp_send_json_success(["request_id" => $result]);
        } elseif (is_array($result) && isset($result["id"])) {
            // API returns object with 'id' field
            wp_send_json_success(["request_id" => $result["id"]]);
        } elseif (is_array($result) && isset($result["request_id"])) {
            // API returns object with 'request_id' field
            wp_send_json_success(["request_id" => $result["request_id"]]);
        } else {
            wp_send_json_error([
                "message" =>
                    "Invalid API response: " . substr($response_body, 0, 200),
            ]);
        }
    }

    /**
     * AJAX: Check status
     */
    public function ajax_check_status()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_die("Unauthorized");
        }

        if (!isset($_POST["request_id"])) {
            wp_send_json_error("No request ID provided");
        }

        $request_id = sanitize_text_field(wp_unslash($_POST["request_id"]));

        if (empty($this->api_key)) {
            wp_send_json_error("API key not configured");
        }

        // Call check status endpoint
        $status_url = $this->api_url . $request_id . "/check_status/";

        $response = wp_remote_get($status_url, [
            "headers" => [
                "X-API-Key" => $this->api_key,
            ],
            "timeout" => 30,
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error("Status check failed");
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code !== 200) {
            wp_send_json_error("Status check error");
        }

        $result = json_decode($response_body, true);

        // If status is completed, fix the image URL
        if (
            isset($result["status"]) &&
            $result["status"] === "completed" &&
            isset($result["output_image_url"])
        ) {
            if (strpos($result["output_image_url"], "http") !== 0) {
                $result["output_image_url"] =
                    "https://danava-ai.ir" . $result["output_image_url"];
            }
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Save image to gallery
     */
    public function ajax_save_to_gallery()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_die("Unauthorized");
        }

        if (!isset($_POST["image_url"])) {
            wp_send_json_error("No image URL provided");
        }

        $image_url = esc_url_raw(wp_unslash($_POST["image_url"]));

        if (empty($image_url)) {
            wp_send_json_error("No image URL provided");
        }

        // Download the image
        $response = wp_remote_get($image_url, [
            "timeout" => 30,
        ]);

        if (
            is_wp_error($response) ||
            wp_remote_retrieve_response_code($response) !== 200
        ) {
            wp_send_json_error("Failed to download image");
        }

        $image_data = wp_remote_retrieve_body($response);
        $filename = basename(wp_parse_url($image_url, PHP_URL_PATH));

        // Prepare for sideload
        $tmp_file = wp_tempnam($filename);
        file_put_contents($tmp_file, $image_data);

        $file_array = [
            "name" => $filename,
            "type" => wp_get_image_mime($tmp_file),
            "tmp_name" => $tmp_file,
            "error" => 0,
            "size" => strlen($image_data),
        ];

        // Sideload the image
        $attachment_id = media_handle_sideload($file_array, 0);

        if (is_wp_error($attachment_id)) {
            wp_send_json_error(
                "Failed to save image to gallery: " .
                    $attachment_id->get_error_message(),
            );
        }

        $attachment_url = wp_get_attachment_url($attachment_id);
        wp_send_json_success([
            "attachment_url" => $attachment_url,
            "attachment_id" => $attachment_id,
        ]);
    }
}
