<?php
/**
 * Danava Gallery Class
 *
 * Handles the image gallery page and functionality.
 */

if (!defined("ABSPATH")) {
    exit();
}

class Danava_Gallery
{
    private $api_key;
    private $api_url;
    private $per_page = 12;

    public function __construct()
    {
        // Initialize when the plugin loads
        add_action("admin_menu", [$this, "add_gallery_page"], 20); // Higher priority number means it runs later
        add_action("admin_init", [$this, "init"]);
        add_action("admin_enqueue_scripts", [$this, "enqueue_scripts"]);
        add_action("wp_ajax_danava_load_gallery", [$this, "ajax_load_gallery"]);
        add_action("wp_ajax_danava_get_image_details", [
            $this,
            "ajax_get_image_details",
        ]);
    }

    public function init()
    {
        $this->api_key = get_option("danava_api_key");
        $this->api_url = rtrim(get_option("danava_api_url"), "/") . "/";

        add_action("admin_menu", [$this, "add_gallery_page"]);
        add_action("admin_enqueue_scripts", [$this, "enqueue_scripts"]);
        add_action("wp_ajax_danava_load_gallery", [$this, "ajax_load_gallery"]);
        add_action("wp_ajax_danava_get_image_details", [
            $this,
            "ajax_get_image_details",
        ]);
    }

    /**
     * Add gallery submenu page
     */
    public function add_gallery_page()
    {
        $parent_slug = "danava-menu";

        $hook = add_submenu_page(
            $parent_slug,
            __("Image Gallery", "danava-image-processor"),
            __("Image Gallery", "danava-image-processor"),
            "manage_options",
            "danava-gallery",
            [$this, "render_gallery_page"],
        );

        return $hook;
    }

    /**
     * Enqueue required scripts and styles
     */
    public function enqueue_scripts($hook)
    {
        if (strpos($hook, "danava-gallery") === false) {
            return;
        }

        wp_enqueue_style(
            "danava-gallery-css",
            plugin_dir_url(dirname(__FILE__)) . "assets/css/gallery.css",
            [],
            "1.0.0",
        );
        // Enqueue SweetAlert2
        wp_enqueue_script(
            "sweetalert2",
            plugin_dir_url(dirname(__FILE__)) . "assets/js/sweetalert2.all.min.js",
            [],
            "11.0.0",
            true,
        );

        wp_enqueue_script(
            "danava-gallery-js",
            plugin_dir_url(dirname(__FILE__)) . "assets/js/gallery.js",
            ["jquery", "sweetalert2"],
            "1.0.0",
            true,
        );

        $localized_data = [
            "ajax_url" => admin_url("admin-ajax.php"),
            "nonce" => wp_create_nonce("danava_ajax_nonce"),
        ];

        wp_localize_script("danava-gallery-js", "danava_ajax", $localized_data);

        // Localize script with translated strings for JavaScript
        wp_localize_script("danava-gallery-js", "danava_i18n", [
            "loading" => __("Loading...", "danava-image-processor"),
            "error" => __("Error", "danava-image-processor"),
            "error_loading_images" => __(
                "Error loading images:",
                "danava-image-processor"
            ),
            "understood" => __("Understood", "danava-image-processor"),
            "connection_error" => __(
                "Connection Error",
                "danava-image-processor"
            ),
            "error_connecting_to_server" => __(
                "Error connecting to server:",
                "danava-image-processor"
            ),
            "load_more" => __("Load More", "danava-image-processor"),
            "error_loading_image_details" => __(
                "Error loading image details",
                "danava-image-processor"
            ),
            "error_connecting_to_server_short" => __(
                "Error connecting to server",
                "danava-image-processor"
            ),
            "saving" => __("Saving...", "danava-image-processor"),
            "success" => __("Success", "danava-image-processor"),
            "image_saved_to_gallery" => __(
                "Image successfully saved to gallery",
                "danava-image-processor"
            ),
            "great" => __("Great", "danava-image-processor"),
            "error_saving_image" => __(
                "Error saving image",
                "danava-image-processor"
            ),
            "save_to_gallery" => __(
                "Save to Gallery",
                "danava-image-processor"
            ),
            "attention" => __("Attention", "danava-image-processor"),
            "use_new_image_page" => __(
                "Please use the new image creation page to add image to product.",
                "danava-image-processor"
            ),
            "description" => __("Description:", "danava-image-processor"),
            "created_date" => __("Created date:", "danava-image-processor"),
            "model" => __("Model:", "danava-image-processor"),
        ]);
    }

    /**
     * Render gallery page
     */
    public function render_gallery_page()
    {
        ?>
        <div class="wrap danava-gallery-wrap">
            <h1><?php esc_html_e("Image Gallery", "danava-image-processor"); ?></h1>

            <div class="danava-gallery-grid" id="danava-gallery-grid">
                <!-- Images will be loaded here via AJAX -->
            </div>

            <div class="danava-gallery-pagination">
                <button class="button button-secondary" id="load-more-images">
                    <?php esc_html_e("Load More", "danava-image-processor"); ?>
                </button>
            </div>

            <!-- Image Details Modal -->
            <div class="danava-modal" id="image-details-modal">
                <div class="danava-modal-content">
                    <span class="danava-modal-close">&times;</span>
                    <div class="danava-modal-body">
                        <div class="image-comparison">
                            <div class="input-images">
                                <h3><?php esc_html_e(
                                    "Input Images",
                                    "danava-image-processor",
                                ); ?></h3>
                                <div class="input-images-container"></div>
                            </div>
                            <div class="output-image">
                                <h3><?php esc_html_e(
                                    "Output Image",
                                    "danava-image-processor",
                                ); ?></h3>
                                <div class="output-image-container"></div>
                            </div>
                        </div>
                        <div class="image-details">
                            <div class="prompt-info"></div>
                            <div class="date-info"></div>
                            <div class="model-info"></div>
                        </div>
                        <div class="action-buttons">
                            <button class="button button-primary save-to-gallery">
                                <?php esc_html_e(
                                    "Save to Gallery",
                                    "danava-image-processor",
                                ); ?>
                            </button>
                            <button class="button button-secondary use-in-product" data-image-url="">
                                <i class="fas fa-shopping-cart"></i>
                                <?php esc_html_e(
                                    "Use in Product",
                                    "danava-image-processor",
                                ); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * AJAX: Load gallery images
     */
    public function ajax_load_gallery()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error("Unauthorized");
        }

        if (empty($this->api_key) || empty($this->api_url)) {
            wp_send_json_error([
                "message" => "API configuration missing",
                "details" => [
                    "api_key_set" => !empty($this->api_key),
                    "api_url_set" => !empty($this->api_url),
                    "api_url_value" => $this->api_url,
                ],
            ]);
        }

        $page = isset($_POST["page"]) ? intval($_POST["page"]) : 1;

        // Get gallery data from API
        $response = wp_remote_get($this->api_url, [
            "headers" => [
                "X-API-Key" => $this->api_key,
            ],
            "sslverify" => false,
            "timeout" => 30,
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error([
                "message" =>
                    "Failed to load gallery: " . $response->get_error_message(),
                "error_type" => "wp_remote_get_error",
                "api_url" => $this->api_url,
            ]);
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        $data = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error("Invalid JSON response from API");
        }

        if (!is_array($data)) {
            wp_send_json_error([
                "message" => "Invalid response from API",
                "response_type" => gettype($data),
                "response_body" => $response_body,
                "json_error" => json_last_error_msg(),
            ]);
        }

        // Calculate pagination
        $total_items = count($data);
        $offset = ($page - 1) * $this->per_page;
        $items = array_slice($data, $offset, $this->per_page);

        wp_send_json_success([
            "items" => $items,
            "has_more" => $offset + $this->per_page < $total_items,
        ]);
    }

    /**
     * AJAX: Get image details
     */
    public function ajax_get_image_details()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error("Unauthorized");
        }

        $id = isset($_POST["id"]) ? intval($_POST["id"]) : 0;

        if (!$id) {
            wp_send_json_error("Invalid image ID");
        }

        // Get image details from API
        $response = wp_remote_get($this->api_url . $id . "/", [
            "headers" => [
                "X-API-Key" => $this->api_key,
            ],
            "sslverify" => false,
            "timeout" => 30,
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error("Failed to load image details");
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!is_array($data)) {
            wp_send_json_error("Invalid response from API");
        }

        wp_send_json_success($data);
    }
}
