<?php
/**
 * Danava Settings Class
 *
 * Handles the plugin settings page and API key configuration.
 */

if (!defined("ABSPATH")) {
    exit();
}

class Danava_Settings
{
    public function __construct()
    {
        add_action("admin_menu", [$this, "add_settings_page"]);
        add_action("admin_init", [$this, "register_settings"]);
        add_action("admin_enqueue_scripts", [$this, "enqueue_scripts"]);
        add_action("wp_ajax_fetch_danava_credit", [
            $this,
            "fetch_credit_balance",
        ]);
    }

    /**
     * Fetch credit balance from Danava API
     */
    public function fetch_credit_balance()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error(["message" => "Unauthorized"]);
        }

        $api_key = get_option("danava_api_key");

        $response = wp_remote_get(
            "https://danava-ai.ir/payment/api/remaining-credits/",
            [
                "headers" => [
                    "x-api-key" => $api_key,
                ],
            ],
        );

        if (is_wp_error($response)) {
            wp_send_json_error(["message" => $response->get_error_message()]);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (isset($data["remaining_credits"])) {
            wp_send_json_success($data);
        } else {
            wp_send_json_error(["message" => "Error fetching credit information"]);
        }
    }

    /**
     * Add Danava menu and submenus
     */
    public function add_settings_page()
    {
        // Add top-level menu
        add_menu_page(
            __("Danava", "danava-image-processor"),
            __("Danava", "danava-image-processor"),
            "manage_options",
            "danava-menu",
            [$this, "settings_page"],
            "dashicons-images-alt2",
            30,
        );

        // Add image editing submenu
        add_submenu_page(
            "danava-menu",
            __("Image Editor", "danava-image-processor"),
            __("Image Editor", "danava-image-processor"),
            "manage_options",
            "danava-image-edit",
            [$this, "image_edit_page"],
        );
    }

    /**
     * Register settings
     */
    public function register_settings()
    {
        register_setting("danava_settings", "danava_api_key", [
            "sanitize_callback" => "sanitize_text_field",
        ]);
        register_setting("danava_settings", "danava_api_url", [
            "sanitize_callback" => "esc_url_raw",
        ]);
        register_setting("danava_settings", "danava_model", [
            "sanitize_callback" => "sanitize_text_field",
        ]);
        register_setting("danava_settings", "danava_prompts", [
            "sanitize_callback" => [$this, "sanitize_prompts"],
        ]);

        add_settings_section(
            "danava_api_section",
            __("API Settings", "danava-image-processor"),
            [$this, "api_section_callback"],
            "danava-settings",
        );

        add_settings_field(
            "danava_api_key",
            __("API Key", "danava-image-processor"),
            [$this, "api_key_field_callback"],
            "danava-settings",
            "danava_api_section",
        );

        add_settings_field(
            "danava_api_url",
            __("API URL", "danava-image-processor"),
            [$this, "api_url_field_callback"],
            "danava-settings",
            "danava_api_section",
        );

        add_settings_field(
            "danava_model",
            __("AI Model", "danava-image-processor"),
            [$this, "model_field_callback"],
            "danava-settings",
            "danava_api_section",
        );

        // Add prompts section
        add_settings_section(
            "danava_prompts_section",
            __("Prompt Management", "danava-image-processor"),
            [$this, "prompts_section_callback"],
            "danava-settings",
        );

        add_settings_field(
            "danava_prompts",
            __("Image Editing Prompts", "danava-image-processor"),
            [$this, "prompts_field_callback"],
            "danava-settings",
            "danava_prompts_section",
        );
    }

    /**
     * Sanitize prompts array
     */
    public function sanitize_prompts($input)
    {
        if (!is_array($input)) {
            return [];
        }

        $sanitized = [];
        foreach ($input as $prompt) {
            if (is_array($prompt) && isset($prompt['label']) && isset($prompt['value'])) {
                $sanitized[] = [
                    'label' => sanitize_text_field($prompt['label']),
                    'value' => sanitize_textarea_field($prompt['value']),
                ];
            }
        }

        return $sanitized;
    }

    /**
     * Settings page content
     */
    public function settings_page()
    {
        ?>
        <div class="wrap danava-settings-wrap">
            <div class="danava-settings-header">
                <div class="header-content">
                    <div class="header-text">
                        <h1 class="settings-title"><?php esc_html_e(
                            "Danava Image Processor Settings",
                            "danava-image-processor",
                        ); ?></h1>
                        <p class="settings-subtitle"><?php esc_html_e(
                            "Configure your Danava AI image processing system",
                            "danava-image-processor",
                        ); ?></p>
                    </div>
                    <div class="header-icon">
                        <i class="fas fa-cog"></i>
                    </div>
                </div>
            </div>

            <!-- Credit Balance Section -->
            <div class="danava-credit-card">
                <div class="credit-header">
                    <div class="credit-icon">
                        <i class="fas fa-coins"></i>
                    </div>
                    <div class="credit-info">
                        <h3><?php esc_html_e(
                            "Account Credit",
                            "danava-image-processor",
                        ); ?></h3>
                        <p><?php esc_html_e(
                            "Manage your credit and purchase processing packages",
                            "danava-image-processor",
                        ); ?></p>
                    </div>
                </div>
                <div class="credit-content">
                    <div class="credit-balance">
                        <div class="balance-label"><?php esc_html_e(
                            "Remaining Credit:",
                            "danava-image-processor",
                        ); ?></div>
                        <div class="balance-amount" id="credit-amount"><?php esc_html_e(
                            "Loading...",
                            "danava-image-processor",
                        ); ?></div>
                    </div>
                    <a href="https://danava-ai.ir/payment/credit_plans/" target="_blank" rel="noopener noreferrer" class="credit-upgrade-btn">
                        <i class="fas fa-plus"></i>
                        <?php esc_html_e("Increase Credit", "danava-image-processor"); ?>
                    </a>
                </div>
            </div>

            <div class="settings-grid">
                <div class="settings-main">
                    <form method="post" action="options.php" class="danava-settings-form">
                        <?php
                        settings_fields("danava_settings");
                        do_settings_sections("danava-settings");
                        ?>
                        <div class="form-actions">
                            <button type="submit" class="btn-save-changes">
                                <i class="fas fa-save"></i>
                                <?php esc_html_e(
                                    "Save Changes",
                                    "danava-image-processor",
                                ); ?>
                            </button>
                        </div>
                    </form>
                </div>

                <div class="settings-sidebar">
                    <div class="info-card">
                        <div class="info-header">
                            <i class="fas fa-info-circle"></i>
                            <h4><?php esc_html_e("Settings Guide", "danava-image-processor"); ?></h4>
                        </div>
                        <div class="info-content">
                            <div class="info-item">
                                <strong><?php esc_html_e("API Key:", "danava-image-processor"); ?></strong>
                                <p><?php esc_html_e("Get your Danava service access key from your user dashboard.", "danava-image-processor"); ?></p>
                            </div>
                            <div class="info-item">
                                <strong><?php esc_html_e("API URL:", "danava-image-processor"); ?></strong>
                                <p><?php esc_html_e("Danava image processing server address (pre-configured by default).", "danava-image-processor"); ?></p>
                            </div>
                            <div class="info-item">
                                <strong><?php esc_html_e("AI Model:", "danava-image-processor"); ?></strong>
                                <p><?php esc_html_e("Select image processing model (currently Gemini is active).", "danava-image-processor"); ?></p>
                            </div>
                        </div>
                    </div>


                </div>
            </div>
        </div>
        <?php
    }

    /**
     * API section callback
     */
    public function api_section_callback()
    {
        echo '<div class="section-header">
                <div class="section-icon">
                    <i class="fas fa-key"></i>
                </div>
                <div class="section-content">
                    <h3>' .
            esc_html__("API Settings", "danava-image-processor") .
            '</h3>
                    <p>' .
            esc_html__(
                "Enter your Danava API information below.",
                "danava-image-processor",
            ) .
            '</p>
                </div>
              </div>';
    }

    /**
     * API key field callback
     */
    public function api_key_field_callback()
    {
        $api_key = get_option("danava_api_key");
        echo '<div class="danava-api-key-field">' .
            '<input type="text" name="danava_api_key" value="' .
            esc_attr($api_key) .
            '" class="regular-text" />' .
            '<a href="https://danava-ai.ir/vendor/api-key/" target="_blank" rel="noopener" class="btn btn-primary mt-5 danava-get-api-btn">' .
            esc_html__("Get API Key", "danava-image-processor") .
            "</a>" .
            "</div>";
    }

    /**
     * API URL field callback
     */
    public function api_url_field_callback()
    {
        $api_url = get_option(
            "danava_api_url",
            "https://danava-ai.ir/ai-support/api/v2/image-generator-requests/",
        );
        echo '<input type="url" name="danava_api_url" value="' .
            esc_attr($api_url) .
            '" class="regular-text" placeholder="https://danava-ai.ir/ai-support/api/v2/image-generator-requests/" />';
    }

    /**
     * Model field callback
     */
    public function model_field_callback()
    {
        $model = get_option("danava_model", "gemini");
        echo '<select name="danava_model">';
        echo '<option value="gemini" ' .
            selected($model, "gemini", false) .
            ">Gemini</option>";
        echo "</select>";
    }

    /**
     * Prompts section callback
     */
    public function prompts_section_callback()
    {
        echo '<div class="section-header">
                <div class="section-icon">
                    <i class="fas fa-edit"></i>
                </div>
                <div class="section-content">
                    <h3>' .
            esc_html__("Prompt Management", "danava-image-processor") .
            '</h3>
                    <p>' .
            esc_html__(
                "Manage preset image editing prompts shown on the editing page.",
                "danava-image-processor",
            ) .
            '</p>
                </div>
              </div>';
    }

    /**
     * Prompts field callback
     */
    public function prompts_field_callback()
    {
        $prompts = get_option("danava_prompts", []);

        // Default prompts if none exist
        if (empty($prompts)) {
            $prompts = [
                [
                    "label" => "Professional Studio",
                    "value" =>
                        "Convert to professional studio photo, decor matching the product, preserve all product details completely, high resolution, harmonious color combinations",
                ],
                [
                    "label" => "Quality Enhancement",
                    "value" =>
                        "Improve image quality with resolution enhancement, reduce digital noise, adjust contrast and brightness, enhance colors using advanced algorithms, preserve fine details",
                ],
                [
                    "label" => "White Background",
                    "value" =>
                        "Change background to pure white with complete removal of background elements, keep product centered in image, adjust lighting to highlight product",
                ],
                [
                    "label" => "Shadow and Depth",
                    "value" =>
                        "Add natural shadows and three-dimensional depth using advanced techniques, create sense of volume and texture, adjust lighting to highlight product features, maintain image realism",
                ],
                [
                    "label" => "Minimal Style",
                    "value" =>
                        "Convert to minimal style with simple background, focus on product, remove unnecessary elements, clean modern design, maintain visual appeal, preserve details",
                ],
                [
                    "label" => "Professional Lighting",
                    "value" =>
                        "Add professional lighting using advanced studio techniques, adjust light intensity, create soft shadows, highlight product features, enhance colors and contrast",
                ],
                [
                    "label" => "Product Zoom",
                    "value" =>
                        "Zoom in on product while maintaining high quality, focus on important details, adjust image composition, improve resolution in zoomed area, maintain overall image balance",
                ],
            ];
        }

        echo '<div id="prompts-management">';
        echo '<div class="prompts-list" id="prompts-list">';

        foreach ($prompts as $index => $prompt) {
            echo '<div class="prompt-item" data-index="' . esc_attr($index) . '">';
            echo '<div class="prompt-fields">';
            echo '<input type="text" name="danava_prompts[' .
                esc_attr($index) .
                '][label]" value="' .
                esc_attr($prompt["label"]) .
                '" placeholder="' . esc_attr__("Suggestion Label", "danava-image-processor") . '" class="regular-text" />';
            echo '<textarea name="danava_prompts[' .
                esc_attr($index) .
                '][value]" rows="2" placeholder="' . esc_attr__("Suggestion Text", "danava-image-processor") . '" class="large-text auto-resize">' .
                esc_textarea($prompt["value"]) .
                "</textarea>";
            echo "</div>";
            echo '<button type="button" class="remove-prompt"></button>';
            echo "</div>";
        }

        echo "</div>";
        echo '<button type="button" class="button button-secondary" id="add-prompt">' . esc_html__("Add New Suggestion", "danava-image-processor") . '</button>';
        echo "</div>";

        // Add JavaScript for dynamic management
        echo '<script>
        jQuery(document).ready(function($) {
            let promptIndex = ' .
            count($prompts) .
            ';

            $("#add-prompt").on("click", function() {
                const promptHtml = `
                    <div class="prompt-item" data-index="` + promptIndex + `">
                        <div class="prompt-fields">
                            <input type="text" name="danava_prompts[` + promptIndex + `][label]" value="" placeholder="` + danava_i18n.suggestion_label + `" class="regular-text" />
                            <textarea name="danava_prompts[` + promptIndex + `][value]" rows="2" placeholder="` + danava_i18n.suggestion_text + `" class="large-text auto-resize"></textarea>
                        </div>
                        <button type="button" class="remove-prompt">` + danava_i18n.remove + `</button>
                    </div>
                `;
                $("#prompts-list").append(promptHtml);
                promptIndex++;
            });

            $(document).on("click", ".remove-prompt", function() {
                $(this).closest(".prompt-item").remove();
            });
        });
        </script>';
    }

    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts($hook)
    {
        if (strpos($hook, "danava") !== false) {
            wp_enqueue_style(
                "font-awesome",
                plugin_dir_url(__FILE__) . "../assets/css/font-awesome.min.css",
                [],
                "6.0.0",
            );
            wp_enqueue_style(
                "danava-admin-css",
                plugin_dir_url(__FILE__) . "../assets/css/admin.css",
                [],
                "1.0.0",
            );

            // Enqueue WooCommerce product selector styles
            if (class_exists("WooCommerce")) {
                wp_enqueue_style(
                    "danava-woocommerce-css",
                    plugin_dir_url(__FILE__) . "../assets/css/woocommerce.css",
                    [],
                    "1.0.0",
                );
            }

            // Enqueue WordPress media scripts for gallery functionality
            wp_enqueue_media();

            // Enqueue SweetAlert2
            wp_enqueue_script(
                "sweetalert2",
                plugin_dir_url(__FILE__) . "../assets/js/sweetalert2.all.min.js",
                [],
                "11.0.0",
                true,
            );

            wp_enqueue_script(
                "danava-admin-js",
                plugin_dir_url(__FILE__) . "../assets/js/admin.js",
                ["jquery", "sweetalert2"],
                "1.0.0",
                true,
            );

            // Enqueue WooCommerce product selector script
            if (class_exists("WooCommerce")) {
                wp_enqueue_script(
                    "danava-woocommerce-js",
                    plugin_dir_url(__FILE__) . "../assets/js/woocommerce.js",
                    ["jquery", "danava-admin-js"],
                    "1.0.0",
                    true,
                );
            }

            wp_localize_script("danava-admin-js", "danava_ajax", [
                "ajax_url" => admin_url("admin-ajax.php"),
                "nonce" => wp_create_nonce("danava_ajax_nonce"),
            ]);

            // Localize script with translated strings for JavaScript
            wp_localize_script("danava-admin-js", "danava_i18n", [
                // Media Gallery
                "select_images" => __("Select Images", "danava-image-processor"),
                "image_limit_title" => __(
                    "Image Limit",
                    "danava-image-processor"
                ),
                "max_3_images" => __(
                    "You cannot select more than 3 images.",
                    "danava-image-processor"
                ),
                "understood" => __("Understood", "danava-image-processor"),
                "too_many_title" => __("Too Many", "danava-image-processor"),
                "max_3_images_detail" => __(
                    "The number of selected images exceeds the limit. You can select a maximum of 3 images.",
                    "danava-image-processor"
                ),

                // Form Validation
                "empty_description" => __(
                    "Empty Description",
                    "danava-image-processor"
                ),
                "enter_description" => __(
                    "Please enter the image description.",
                    "danava-image-processor"
                ),
                "no_image_selected" => __(
                    "No Image Selected",
                    "danava-image-processor"
                ),
                "select_at_least_one" => __(
                    "Please select at least one image.",
                    "danava-image-processor"
                ),

                // Processing Status
                "sending" => __("Sending...", "danava-image-processor"),
                "error" => __("Error", "danava-image-processor"),
                "error_sending_request" => __(
                    "Error sending request",
                    "danava-image-processor"
                ),
                "connection_error" => __(
                    "Connection Error",
                    "danava-image-processor"
                ),
                "start_image_generation" => __(
                    "Start Image Generation",
                    "danava-image-processor"
                ),
                "processing" => __("Processing...", "danava-image-processor"),
                "generating_image" => __(
                    "Generating image...",
                    "danava-image-processor"
                ),
                "please_wait" => __(
                    "Please wait until the process completes",
                    "danava-image-processor"
                ),
                "back" => __("Back", "danava-image-processor"),

                // Save to Gallery
                "saving" => __("Saving...", "danava-image-processor"),
                "success" => __("Success", "danava-image-processor"),
                "image_saved_to_gallery" => __(
                    "Image successfully saved to gallery.",
                    "danava-image-processor"
                ),
                "great" => __("Great", "danava-image-processor"),
                "saved" => __("Saved", "danava-image-processor"),
                "error_saving_image" => __(
                    "Error saving image",
                    "danava-image-processor"
                ),
                "save_to_gallery" => __(
                    "Save to Gallery",
                    "danava-image-processor"
                ),

                // Completed Status
                "image_generated_success" => __(
                    "Image generated successfully!",
                    "danava-image-processor"
                ),
                "use_in_product" => __(
                    "Use in Product",
                    "danava-image-processor"
                ),
                "new_request" => __("New Request", "danava-image-processor"),

                // Failed Status
                "error_generating_image" => __(
                    "Error generating image",
                    "danava-image-processor"
                ),
                "unknown_error" => __(
                    "Unknown error",
                    "danava-image-processor"
                ),
                "try_again" => __("Try Again", "danava-image-processor"),

                // Product Selection
                "select_product" => __(
                    "Select Product",
                    "danava-image-processor"
                ),
                "search_product" => __(
                    "Search product...",
                    "danava-image-processor"
                ),
                "loading_products" => __(
                    "Loading products...",
                    "danava-image-processor"
                ),
                "no_product_found" => __(
                    "No product found",
                    "danava-image-processor"
                ),
                "error_loading_products" => __(
                    "Error loading products",
                    "danava-image-processor"
                ),
                "select" => __("Select", "danava-image-processor"),
                "images_text" => __("images", "danava-image-processor"),

                // Product Images
                "loading_product_images" => __(
                    "Loading product images...",
                    "danava-image-processor"
                ),
                "error_loading_product_images" => __(
                    "Error loading product images",
                    "danava-image-processor"
                ),
                "back_to_products" => __(
                    "Back to products list",
                    "danava-image-processor"
                ),
                "new_image" => __("New image:", "danava-image-processor"),
                "how_to_use_image" => __(
                    "How to use image:",
                    "danava-image-processor"
                ),
                "set_as_main" => __(
                    "Set as main product image",
                    "danava-image-processor"
                ),
                "add_to_gallery" => __(
                    "Add to product gallery",
                    "danava-image-processor"
                ),
                "set_as_main_and_gallery" => __(
                    "Set as main image and add to gallery",
                    "danava-image-processor"
                ),
                "existing_product_images" => __(
                    "Existing product images:",
                    "danava-image-processor"
                ),
                "reorder_info" => __(
                    "Use up/down buttons to change image order",
                    "danava-image-processor"
                ),
                "confirm_and_apply" => __(
                    "Confirm and Apply Image",
                    "danava-image-processor"
                ),
                "applying" => __("Applying...", "danava-image-processor"),
                "image_added_to_product" => __(
                    "Image successfully added to product!",
                    "danava-image-processor"
                ),
                "view_product" => __(
                    "View Product",
                    "danava-image-processor"
                ),
                "view_product_question" => __(
                    "Do you want to view the product?",
                    "danava-image-processor"
                ),
                "yes_view" => __("Yes, view it", "danava-image-processor"),
                "no" => __("No", "danava-image-processor"),
                "error_applying_image" => __(
                    "Error applying image to product",
                    "danava-image-processor"
                ),
                "main_image_badge" => __(
                    "Main image",
                    "danava-image-processor"
                ),

                // WooCommerce specific
                "selected_badge" => __("Selected", "danava-image-processor"),
                "no_images_found" => __(
                    "No image found",
                    "danava-image-processor"
                ),
                "product_no_images" => __(
                    "This product has no images.",
                    "danava-image-processor"
                ),
                "main_badge" => __("Main", "danava-image-processor"),
                "max_3_remove_first" => __(
                    "Maximum 3 images can be selected. Please remove one first.",
                    "danava-image-processor"
                ),
                "images_added_success" => __(
                    "Images added successfully!",
                    "danava-image-processor"
                ),
                "of" => __("of", "danava-image-processor"),
                "from" => __("from", "danava-image-processor"),
                "image_selected" => __(
                    "image selected",
                    "danava-image-processor"
                ),

                // Prompt Management
                "suggestion_label" => __("Suggestion Label", "danava-image-processor"),
                "suggestion_text" => __("Suggestion Text", "danava-image-processor"),
                "remove" => __("Remove", "danava-image-processor"),
            ]);

            // Add credit balance fetch script
            $error_message = esc_js(__("Error fetching data", "danava-image-processor"));
            wp_add_inline_script(
                "danava-admin-js",
                "
                jQuery(document).ready(function($) {
                    function fetchCreditBalance() {
                        $.ajax({
                            url: danava_ajax.ajax_url,
                            type: 'POST',
                            data: {
                                action: 'fetch_danava_credit',
                                nonce: danava_ajax.nonce
                            },
                            success: function(response) {
                                if (response.success && response.data.remaining_credits !== undefined) {
                                    $('#credit-amount').text(response.data.remaining_credits.toLocaleString());
                                } else {
                                    $('#credit-amount').text('" . $error_message . "');
                                }
                            },
                            error: function() {
                                $('#credit-amount').text('" . $error_message . "');
                            }
                        });
                    }

                    // Fetch credit balance when the page loads
                    fetchCreditBalance();

                    // Refresh credit balance every 5 minutes
                    setInterval(fetchCreditBalance, 300000);
                });
            "
            );
        }
    }

    /**
     * Image editing page
     */
    public function image_edit_page()
    {
        $template = __DIR__ . "/views/image-edit-page.php";
        if (file_exists($template)) {
            include $template;
        } else {
            echo '<div class="notice notice-error"><p>Template not found: includes/views/image-edit-page.php</p></div>';
        }
    }
}
