<?php
/**
 * Danava WooCommerce Integration Class
 *
 * Handles WooCommerce product selection and image retrieval.
 */

if (!defined("ABSPATH")) {
    exit();
}

class Danava_WooCommerce
{
    public function __construct()
    {
        // Check if WooCommerce is active
        if (!class_exists("WooCommerce")) {
            return;
        }

        add_action("wp_ajax_danava_search_products", [
            $this,
            "ajax_search_products",
        ]);
        add_action("wp_ajax_danava_get_product_images", [
            $this,
            "ajax_get_product_images",
        ]);
        add_action("wp_ajax_danava_apply_image_to_product", [
            $this,
            "ajax_apply_image_to_product",
        ]);
    }

    /**
     * Check if WooCommerce is active
     */
    public static function is_woocommerce_active()
    {
        return class_exists("WooCommerce");
    }

    /**
     * AJAX: Search WooCommerce products
     */
    public function ajax_search_products()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error("Unauthorized");
        }

        $search = isset($_POST["search"])
            ? sanitize_text_field(wp_unslash($_POST["search"]))
            : "";
        $page = isset($_POST["page"]) ? intval($_POST["page"]) : 1;
        $per_page = 20;

        $args = [
            "post_type" => "product",
            "posts_per_page" => $per_page,
            "paged" => $page,
            "post_status" => "publish",
            "orderby" => "title",
            "order" => "ASC",
        ];

        if (!empty($search)) {
            $args["s"] = $search;
        }

        $query = new WP_Query($args);
        $products = [];

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $product_id = get_the_ID();
                $product = wc_get_product($product_id);

                if (!$product) {
                    continue;
                }

                // Get product thumbnail
                $thumbnail_id = $product->get_image_id();
                $thumbnail_url = "";

                if ($thumbnail_id) {
                    $thumbnail_url = wp_get_attachment_image_url(
                        $thumbnail_id,
                        "thumbnail",
                    );
                }

                // Count total images (featured + gallery)
                $gallery_ids = $product->get_gallery_image_ids();
                $total_images = ($thumbnail_id ? 1 : 0) + count($gallery_ids);

                $products[] = [
                    "id" => $product_id,
                    "title" => get_the_title(),
                    "thumbnail" => $thumbnail_url,
                    "price" => $product->get_price_html(),
                    "sku" => $product->get_sku(),
                    "total_images" => $total_images,
                    "permalink" => get_permalink($product_id),
                ];
            }
        }

        wp_reset_postdata();

        wp_send_json_success([
            "products" => $products,
            "has_more" => $page < $query->max_num_pages,
            "total_pages" => $query->max_num_pages,
            "total_products" => $query->found_posts,
        ]);
    }

    /**
     * AJAX: Get product images
     */
    public function ajax_get_product_images()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error("Unauthorized");
        }

        $product_id = isset($_POST["product_id"])
            ? intval($_POST["product_id"])
            : 0;

        if (!$product_id) {
            wp_send_json_error("Invalid product ID");
        }

        $product = wc_get_product($product_id);

        if (!$product) {
            wp_send_json_error("Product not found");
        }

        $images = [];

        // Get featured image
        $featured_image_id = $product->get_image_id();
        if ($featured_image_id) {
            $full_url = wp_get_attachment_image_url($featured_image_id, "full");
            $medium_url = wp_get_attachment_image_url(
                $featured_image_id,
                "medium",
            );
            $thumbnail_url = wp_get_attachment_image_url(
                $featured_image_id,
                "thumbnail",
            );

            $images[] = [
                "id" => $featured_image_id,
                "url" => $full_url,
                "medium" => $medium_url,
                "thumbnail" => $thumbnail_url,
                "title" => get_the_title($featured_image_id),
                "alt" => get_post_meta(
                    $featured_image_id,
                    "_wp_attachment_image_alt",
                    true,
                ),
                "is_featured" => true,
            ];
        }

        // Get gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ($gallery_ids as $image_id) {
            $full_url = wp_get_attachment_image_url($image_id, "full");
            $medium_url = wp_get_attachment_image_url($image_id, "medium");
            $thumbnail_url = wp_get_attachment_image_url(
                $image_id,
                "thumbnail",
            );

            $images[] = [
                "id" => $image_id,
                "url" => $full_url,
                "medium" => $medium_url,
                "thumbnail" => $thumbnail_url,
                "title" => get_the_title($image_id),
                "alt" => get_post_meta(
                    $image_id,
                    "_wp_attachment_image_alt",
                    true,
                ),
                "is_featured" => false,
            ];
        }

        wp_send_json_success([
            "product_id" => $product_id,
            "product_title" => $product->get_name(),
            "images" => $images,
            "total_images" => count($images),
        ]);
    }

    /**
     * Get product by ID
     */
    public static function get_product($product_id)
    {
        if (!self::is_woocommerce_active()) {
            return null;
        }

        return wc_get_product($product_id);
    }

    /**
     * Get all product images
     */
    public static function get_product_images($product_id)
    {
        if (!self::is_woocommerce_active()) {
            return [];
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            return [];
        }

        $images = [];

        // Featured image
        $featured_image_id = $product->get_image_id();
        if ($featured_image_id) {
            $images[] = wp_get_attachment_url($featured_image_id);
        }

        // Gallery images
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ($gallery_ids as $image_id) {
            $images[] = wp_get_attachment_url($image_id);
        }

        return $images;
    }

    /**
     * AJAX: Apply image to WooCommerce product
     */
    public function ajax_apply_image_to_product()
    {
        check_ajax_referer("danava_ajax_nonce", "nonce");

        if (!current_user_can("manage_options")) {
            wp_send_json_error("Unauthorized");
        }

        $product_id = isset($_POST["product_id"])
            ? intval($_POST["product_id"])
            : 0;
        $image_url = isset($_POST["image_url"])
            ? esc_url_raw(wp_unslash($_POST["image_url"]))
            : "";
        $image_action = isset($_POST["image_action"])
            ? sanitize_text_field(wp_unslash($_POST["image_action"]))
            : "main";
        $gallery_order = isset($_POST["gallery_order"])
            ? json_decode(sanitize_text_field(wp_unslash($_POST["gallery_order"])), true)
            : null;

        if (!$product_id || !$image_url) {
            wp_send_json_error("Invalid parameters");
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error("Product not found");
        }

        // Download and save the image to WordPress media library
        $attachment_id = $this->save_image_to_media_library(
            $image_url,
            $product_id,
        );

        if (is_wp_error($attachment_id)) {
            wp_send_json_error(
                "Failed to save image: " . $attachment_id->get_error_message(),
            );
        }

        // Get current gallery images
        $current_gallery_ids = $product->get_gallery_image_ids();

        // Apply image based on selected action
        switch ($image_action) {
            case "main":
                // Set as featured image only
                set_post_thumbnail($product_id, $attachment_id);

                // Update gallery order if provided (without adding new image)
                if (!empty($gallery_order) && is_array($gallery_order)) {
                    $product->set_gallery_image_ids($gallery_order);
                    $product->save();
                }
                break;

            case "gallery":
                // Add to gallery only
                if (!empty($gallery_order) && is_array($gallery_order)) {
                    // User reordered existing gallery, add new image at the end
                    $new_gallery_ids = array_merge($gallery_order, [
                        $attachment_id,
                    ]);
                } else {
                    // No reordering, just append to existing gallery
                    $new_gallery_ids = $current_gallery_ids;
                    $new_gallery_ids[] = $attachment_id;
                }

                $product->set_gallery_image_ids($new_gallery_ids);
                $product->save();
                break;

            case "both":
                // Set as featured image AND add to gallery
                set_post_thumbnail($product_id, $attachment_id);

                if (!empty($gallery_order) && is_array($gallery_order)) {
                    // User reordered existing gallery, add new image at the end
                    $new_gallery_ids = array_merge($gallery_order, [
                        $attachment_id,
                    ]);
                } else {
                    // No reordering, just append to existing gallery
                    $new_gallery_ids = $current_gallery_ids;
                    $new_gallery_ids[] = $attachment_id;
                }

                $product->set_gallery_image_ids($new_gallery_ids);
                $product->save();
                break;
        }

        // Clear product cache
        wc_delete_product_transients($product_id);

        // Get product edit URL
        $product_edit_url = admin_url(
            "post.php?post=" . $product_id . "&action=edit",
        );

        wp_send_json_success([
            "message" => "Image successfully applied to product",
            "attachment_id" => $attachment_id,
            "product_id" => $product_id,
            "product_url" => $product_edit_url,
        ]);
    }

    /**
     * Save remote image to WordPress media library
     */
    private function save_image_to_media_library($image_url, $product_id = 0)
    {
        require_once ABSPATH . "wp-admin/includes/file.php";
        require_once ABSPATH . "wp-admin/includes/media.php";
        require_once ABSPATH . "wp-admin/includes/image.php";

        // Download image to temp file
        $temp_file = download_url($image_url);

        if (is_wp_error($temp_file)) {
            return $temp_file;
        }

        // Prepare file array
        $file_array = [
            "name" => "danava-generated-" . time() . ".png",
            "tmp_name" => $temp_file,
        ];

        // Check file type
        $file_type = wp_check_filetype($file_array["name"]);
        if (!$file_type["type"]) {
            wp_delete_file($file_array["tmp_name"]);
            return new WP_Error("invalid_file", "Invalid file type");
        }

        // Upload to media library
        $attachment_id = media_handle_sideload(
            $file_array,
            $product_id,
            "Danava Generated Image",
        );

        // Clean up temp file if it still exists
        if (file_exists($temp_file)) {
            wp_delete_file($temp_file);
        }

        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        // Set alt text
        update_post_meta(
            $attachment_id,
            "_wp_attachment_image_alt",
            "Danava Generated Image",
        );

        return $attachment_id;
    }
}
